<?php
/**
 * Enviar story para renderização no Cloudflare (ASSÍNCRONO)
 */
function vasb_send_to_cloudflare_renderer($post_id) {
    // Verificar se é uma story e se está publicada/atualizada
    if (get_post_type($post_id) !== 'vasb_story' || wp_is_post_revision($post_id)) {
        return;
    }

    // Verificar licença APENAS para stories novas
    $optimized_html = get_post_meta($post_id, '_vasb_optimized_html', true);
    $license_data = vasb_check_license_status();
    
    // Se já tem HTML otimizado (story antiga), permite reotimização SEM licença
    if (!$optimized_html && !$license_data) {
        error_log('VASB PRO: Licença não ativa para NOVA story - Post ID: ' . $post_id);
        return;
    }

    // Agendar renderização em background para não travar o WordPress
    wp_schedule_single_event(time() + 2, 'vasb_background_render', [$post_id]);
}
add_action('save_post_vasb_story', 'vasb_send_to_cloudflare_renderer', 20);

/**
 * Renderização em background
 */
function vasb_background_render($post_id) {
    // Coletar dados da story
    $story_data = vasb_prepare_story_data($post_id);
    
    // Enviar para Cloudflare Worker
    $response = wp_remote_post(VASB_LICENSE_API_URL . '/render', [
        'headers' => [
            'Content-Type' => 'application/json',
        ],
        'body' => json_encode($story_data),
        'timeout' => 30,
        'sslverify' => false
    ]);

    if (is_wp_error($response)) {
        error_log('VASB PRO: Erro ao enviar para Cloudflare: ' . $response->get_error_message());
        return false;
    }

    $optimized_html = wp_remote_retrieve_body($response);
    $response_code = wp_remote_retrieve_response_code($response);
    
    error_log('VASB PRO: Resposta do Worker - Código: ' . $response_code . ', Tamanho: ' . strlen($optimized_html));
    
    // VALIDAR AMP ANTES DE SALVAR
    if (!empty($optimized_html) && strlen($optimized_html) > 100 && $response_code === 200) {
        
        // Validar estrutura AMP
        if (!vasb_validate_amp_structure($optimized_html)) {
            error_log('VASB PRO: HTML AMP inválido - Post ID: ' . $post_id);
            
            // Criar HTML de fallback para debug
            $debug_html = vasb_create_debug_amp_story($post_id, $story_data);
            update_post_meta($post_id, '_vasb_optimized_html', $debug_html);
            update_post_meta($post_id, '_vasb_amp_validation_error', 'Estrutura AMP inválida');
            
            return false;
        }
        
        // Salvar HTML válido
        update_post_meta($post_id, '_vasb_optimized_html', $optimized_html);
        update_post_meta($post_id, '_vasb_last_optimized', time());
        update_post_meta($post_id, '_vasb_optimized_version', '1.0.0');
        delete_post_meta($post_id, '_vasb_amp_validation_error'); // Limpar erro anterior
        
        error_log('VASB PRO: Story renderizada com sucesso - Post ID: ' . $post_id);
        return true;
    } else {
        error_log('VASB PRO: HTML renderizado vazio ou inválido - Post ID: ' . $post_id);
        return false;
    }
}
add_action('vasb_background_render', 'vasb_background_render');

/**
 * Obter imagem para poster-portrait-src com fallback inteligente
 */
function vasb_get_poster_portrait_src($post_id, $content, $publisher_logo_url) {
    // 1. PRIMEIRA OPÇÃO: Imagem destacada do post
    if (has_post_thumbnail($post_id)) {
        $featured_image = wp_get_attachment_image_url(get_post_thumbnail_id($post_id), 'full');
        if ($featured_image) {
            error_log('VASB: Usando imagem destacada como poster: ' . $featured_image);
            return $featured_image;
        }
    }
    
    // 2. SEGUNDA OPÇÃO: Primeira imagem do conteúdo
    preg_match_all('/\((\d+)\)/', $content, $image_matches);
    if (!empty($image_matches[1])) {
        $first_image_id = intval($image_matches[1][0]);
        $first_image_url = wp_get_attachment_url($first_image_id);
        if ($first_image_url) {
            error_log('VASB: Usando primeira imagem do conteúdo como poster: ' . $first_image_url);
            return $first_image_url;
        }
    }
    
    // 3. TERCEIRA OPÇÃO: Logo do publisher (fallback)
    if (!empty($publisher_logo_url)) {
        error_log('VASB: Usando logo do publisher como poster fallback: ' . $publisher_logo_url);
        return $publisher_logo_url;
    }

    // 4. Se não tiver nada, retorna string vazia
    error_log('VASB: Nenhuma imagem encontrada para poster-portrait-src');
    return '';
}

/**
 * Preparar dados da story para envio - ATUALIZADA para incluir license_key
 */
function vasb_prepare_story_data($post_id) {
    $global_options = get_option(VASB_OPTIONS_NAME, []);
    $content = get_post_meta($post_id, '_vasb_pages_content_raw', true);
    
    // EXTRAIR TODOS OS IDs DE MÍDIA DO CONTEÚDO
    $media_ids = [];
    
    // Buscar IDs de imagens: (123)
    preg_match_all('/\((\d+)\)/', $content, $image_matches);
    if (!empty($image_matches[1])) {
        $media_ids = array_merge($media_ids, $image_matches[1]);
    }
    
    // Buscar IDs de vídeos: (video:123)
    preg_match_all('/\(video\s*:\s*(\d+)/', $content, $video_matches);
    if (!empty($video_matches[1])) {
        $media_ids = array_merge($media_ids, $video_matches[1]);
    }
    
    // Buscar IDs de posters: (poster:456)
    preg_match_all('/poster\s*:\s*(\d+)/', $content, $poster_matches);
    if (!empty($poster_matches[1])) {
        $media_ids = array_merge($media_ids, $poster_matches[1]);
    }
    
    // OBTER URLs DAS MÍDIAS
    $media_urls = [];
    foreach (array_unique($media_ids) as $media_id) {
        $url = wp_get_attachment_url($media_id);
        if ($url) {
            $media_urls[$media_id] = $url;
        }
    }
    
    // Obter URL da logo do publisher
    $publisher_logo_url = '';
    $publisher_logo_id = isset($global_options['publisher_logo_id']) ? intval($global_options['publisher_logo_id']) : 0;
    if ($publisher_logo_id > 0) {
        $publisher_logo_url = wp_get_attachment_image_url($publisher_logo_id, 'full');
    }
    
    // Obter imagem para poster-portrait-src
    $poster_portrait_src = vasb_get_poster_portrait_src($post_id, $content, $publisher_logo_url);
    
    // CRÍTICO: Obter license_key do banco
    $license_data = get_option(VASB_LICENSE_KEY_OPTION);
    $license_key = $license_data ? $license_data['license_key'] : '';

    return array(
        'post_id' => $post_id,
        'title' => get_the_title($post_id),
        'excerpt' => get_the_excerpt($post_id),
        'content' => $content,
        'styles' => array(
            'page_bg_color' => get_post_meta($post_id, '_vasb_page_bg_color', true),
            'page_text_color' => get_post_meta($post_id, '_vasb_page_text_color', true),
            'font_family' => get_post_meta($post_id, '_vasb_font_family', true),
            'page_font_size' => get_post_meta($post_id, '_vasb_page_font_size', true),
            'bg_density' => get_post_meta($post_id, '_vasb_bg_density', true),
            'text_position' => get_post_meta($post_id, '_vasb_text_position', true),
            'text_bg_style' => get_post_meta($post_id, '_vasb_text_bg_style', true),
        ),
        'global_settings' => array(
            'publisher_name' => isset($global_options['publisher_name']) ? $global_options['publisher_name'] : '',
            'publisher_logo_url' => $publisher_logo_url,
            'adsense_publisher_id' => isset($global_options['adsense_publisher_id']) ? $global_options['adsense_publisher_id'] : '',
            'adsense_slot_id' => isset($global_options['adsense_slot_id']) ? $global_options['adsense_slot_id'] : '',
            'analytics_id' => isset($global_options['analytics_id']) ? $global_options['analytics_id'] : '',
        ),
        'site_info' => array(
            'url' => get_permalink($post_id),
            'name' => get_bloginfo('name')
        ),
        'media_urls' => $media_urls,
        'poster_portrait_src' => $poster_portrait_src,
        'license_key' => $license_key,
        'timestamp' => time()
    );
}

/**
 * Validar estrutura AMP antes de salvar
 */
function vasb_validate_amp_structure($html) {
    // Verificar atributos obrigatórios do amp-story
    $required_attributes = array(
        'standalone' => '',
        'title' => '',
        'publisher' => '', 
        'publisher-logo-src' => '',
        'poster-portrait-src' => ''
    );
    
    // Verificar se todas as tags necessárias existem
    $required_tags = array(
        '<amp-story',
        '<amp-story-page',
        '<amp-story-grid-layer',
        '</amp-story>'
    );
    
    foreach ($required_tags as $tag) {
        if (strpos($html, $tag) === false) {
            error_log('VASB PRO: Tag AMP obrigatória faltando: ' . $tag);
            return false;
        }
    }
    
    // Verificar se publisher-logo-src é uma URL válida
    if (preg_match('/publisher-logo-src="([^"]*)"/', $html, $matches)) {
        $logo_src = $matches[1];
        if (empty($logo_src) || filter_var($logo_src, FILTER_VALIDATE_URL) === false) {
            error_log('VASB PRO: publisher-logo-src inválido: ' . $logo_src);
            return false;
        }
    } else {
        error_log('VASB PRO: publisher-logo-src não encontrado');
        return false;
    }

    // Verificar se poster-portrait-src é uma URL válida
    if (preg_match('/poster-portrait-src="([^"]*)"/', $html, $matches)) {
        $poster_src = $matches[1];
        if (empty($poster_src) || filter_var($poster_src, FILTER_VALIDATE_URL) === false) {
            error_log('VASB PRO: poster-portrait-src inválido: ' . $poster_src);
            return false;
        }
    } else {
        error_log('VASB PRO: poster-portrait-src não encontrado');
        return false;
    }
    
    return true;
}

/**
 * Criar story AMP básica para debug quando o Cloudflare falhar
 */
function vasb_create_debug_amp_story($post_id, $story_data) {
    $global_settings = $story_data['global_settings'];
    
    $html = '<!DOCTYPE html>
<html amp lang="pt-BR">
<head>
    <meta charset="utf-8">
    <script async src="https://cdn.ampproject.org/v0.js"></script>
    <script async custom-element="amp-story" src="https://cdn.ampproject.org/v0/amp-story-1.0.js"></script>
    <title>' . esc_html($story_data['title']) . '</title>
    <link rel="canonical" href="' . get_permalink($post_id) . '">
    <meta name="viewport" content="width=device-width,minimum-scale=1,initial-scale=1">
    <style amp-boilerplate>body{-webkit-animation:-amp-start 8s steps(1,end) 0s 1 normal both;-moz-animation:-amp-start 8s steps(1,end) 0s 1 normal both;-ms-animation:-amp-start 8s steps(1,end) 0s 1 normal both;animation:-amp-start 8s steps(1,end) 0s 1 normal both}@-webkit-keyframes -amp-start{from{visibility:hidden}to{visibility:visible}}@-moz-keyframes -amp-start{from{visibility:hidden}to{visibility:visible}}@-ms-keyframes -amp-start{from{visibility:hidden}to{visibility:visible}}@-o-keyframes -amp-start{from{visibility:hidden}to{visibility:visible}}@keyframes -amp-start{from{visibility:hidden}to{visibility:visible}}</style><noscript><style amp-boilerplate>body{-webkit-animation:none;-moz-animation:none;-ms-animation:none;animation:none}</style></noscript>
</head>
<body>
    <amp-story standalone 
        title="' . esc_attr($story_data['title']) . '" 
        publisher="' . esc_attr($global_settings['publisher_name']) . '" 
        publisher-logo-src="' . esc_url($global_settings['publisher_logo_url']) . '" 
        poster-portrait-src="' . esc_url($global_settings['publisher_logo_url']) . '">
        
        <amp-story-page id="cover">
            <amp-story-grid-layer template="vertical">
                <h1>' . esc_html($story_data['title']) . '</h1>
                <p>Story em processo de renderização...</p>
            </amp-story-grid-layer>
        </amp-story-page>
        
    </amp-story>
</body>
</html>';
    
    return $html;
}

// AJAX para re-renderizar story
function vasb_ajax_reoptimize_story() {
    check_ajax_referer('vasb_reoptimize_nonce', 'nonce');
    
    $post_id = intval($_POST['post_id']);
    $success = vasb_background_render($post_id);
    
    if ($success) {
        wp_send_json_success(array('message' => 'Story re-renderizada com sucesso!'));
    } else {
        wp_send_json_error(array('message' => 'Erro ao re-renderizar story.'));
    }
}
add_action('wp_ajax_vasb_reoptimize_story', 'vasb_ajax_reoptimize_story');